// Expected global variables:
/*global consoleEvents consoleUtils domHelper htmlTreeHelpers mainBrowser onErrorHandler remoteHelpers styleHelper toolUI */

// Allow 'eval' for debugging code execution and leading underscores for invoke functions
/*jshint evil: true, nomen: false*/

var messageHandlers;
var remoteCode = {

    // A constant that defines the max number of items to traverse in an expandable tree object for each snapshot,
    // This number will be tweaked as development continues.
    maxItemTraverseCount: 500,

    // The queue of notification messages that have been received before the console is ready
    consoleNotificationQueue: [],

    // The current window that we are executing code against that can be changed with the cd command
    currentWindowContext: null,

    // A mapped set of stored results
    resultMap: {},

    initialize: function console$remoteCode$initialize() {
        /// <summary>
        ///     Gets the remote script code ready for use. 
        ///     Creates the console object and adds it to the page under diagnostic debugging and also sets up communication
        ///     with the VS side console
        /// </summary>

        // Listen for the before script execute event so we can insert our console object
        mainBrowser.addEventListener("BeforeScriptExecute", remoteCode.onBeforeScriptExecute);

        // Try to listen to console messages from IE
        try {
            consoleEvents.addEventListener("message", remoteCode.onConsoleMessage);
        } catch (e) {
            // Do nothing, we're on a version of IE that doesn't support console logging.
        }

        // Create a port
        remoteHelpers.initialize("ConsolePort", remoteCode.initializePage, "__VISUALSTUDIO_CONSOLE_BREAKMODE_FUNC");
    },

    initializePage: function console$remoteCode$initializePage() {
        /// <summary>
        ///     This method informs the Console that the remote side is ready
        ///     It sends the message over the communication port
        /// </summary>

        try {
            remoteCode.initializeConsoles(mainBrowser);
            remoteCode.currentWindowContext = mainBrowser.document.parentWindow;

            var connectionInfo = { docMode: mainBrowser.document.documentMode, contextInfo: mainBrowser.document.parentWindow.location.href };
            remoteHelpers.port.postMessage("Handshake:" + JSON.stringify(connectionInfo));

            // Reset the document timeout
            remoteHelpers.initializeDocumentTries = 0;
        } catch (e) {
            remoteCode.onDocumentNotReady();
        }
    },

    onDocumentNotReady: function console$remoteCode$onDocumentNotReady() {
        /// <summary>
        ///     This method should be called when the document is not yet ready for diagnostics to attach and populate.
        ///     It will increment the timeout counter and post back to the VS side asking for it to re-try the initialize call.
        /// </summary>

        if (remoteHelpers.initializeDocumentTries < remoteHelpers.initializeDocumentMaxTries) {
            remoteHelpers.initializeDocumentTries++;
            remoteHelpers.port.postMessage("DocumentNotYetReady");
        } else {
            onErrorHandler("Document timed out", "remote.js", 0);
        }
    },

    initializeConsoles: function console$remoteCode$initializeConsoles(windowToAttach) {
        /// <summary>
        ///     Recurses through all the frames on the window and calls 
        ///     'initializeConsole' to intialize the console on that frame
        /// </summary>
        /// <param name="windowToAttach" type="Object">
        ///     The IDispatch window that we should inspect for frames
        /// </param>

        var realWindow = null;

        try {
            // Try to get the window object that javascript expects
            realWindow = windowToAttach.document.parentWindow;
        } catch (ex) {
            // Ignore this beforeScriptExecute, as the window is not valid and cannot attach a console
            return;
        }

        // Attach a console to this window
        remoteCode.initializeConsole(realWindow);

        // Use recursion to add a console to each frame
        if (realWindow.frames) {
            for (var i = 0; i < realWindow.frames.length; i++) {
                var frame = realWindow.frames[i];
                if (frame) {
                    var iframe = domHelper.getCrossSiteWindow(realWindow, frame);
                    remoteCode.initializeConsoles(iframe);
                }
            }
        }
    },

    initializeConsole: function console$remoteCode$initializeConsole(realWindow) {
        /// <summary>
        ///     Initializes the console for the specified window
        /// </summary>
        /// <param name="realWindow" type="Object">
        ///     The window object that we add the console to
        /// </param>

        var modifiedConsole = false;
        var consoleObj = realWindow.console;
        var injectedConsole = {};

        if (!consoleObj) {
            // Create a new console object
            consoleObj = realWindow.console = {};
            modifiedConsole = true;
        }

        // The default logging functions will write the messages to our console, so we only define them if needed
        if (consoleObj.log === undefined) {
            consoleObj.log = function (msg) {
                remoteCode.onConsoleFunc(consoleUtils.consoleNotifyType.log, msg);
            };
            injectedConsole.log = consoleObj.log;
            modifiedConsole = true;
        }
        if (consoleObj.info === undefined) {
            consoleObj.info = function (msg) {
                remoteCode.onConsoleFunc(consoleUtils.consoleNotifyType.info, msg);
            };
            injectedConsole.info = consoleObj.info;
            modifiedConsole = true;
        }
        if (consoleObj.warn === undefined) {
            consoleObj.warn = function (msg) {
                remoteCode.onConsoleFunc(consoleUtils.consoleNotifyType.warn, msg);
            };
            injectedConsole.warn = consoleObj.warn;
            modifiedConsole = true;
        }
        if (consoleObj.error === undefined) {
            consoleObj.error = function (msg) {
                remoteCode.onConsoleFunc(consoleUtils.consoleNotifyType.error, msg);
            };
            injectedConsole.error = consoleObj.error;
            modifiedConsole = true;
        }
        if (consoleObj.assert === undefined) {
            consoleObj.assert = function (test, msg) {
                if (!test) {
                    remoteCode.onConsoleFunc(consoleUtils.consoleNotifyType.assert, msg);
                }
            };
            injectedConsole.assert = consoleObj.assert;
            modifiedConsole = true;
        }
        
        // Clear and Dir functions will have default 'no-op' functions, so we need to replace those if they are the native implementations
        if (consoleObj.clear === undefined || (consoleObj.clear && consoleObj.clear.toString && consoleObj.clear.toString() === "\nfunction clear() {\n    [native code]\n}\n")) {
            consoleObj.clear = function () {
                remoteCode.onConsoleFunc("clear");
            };
            injectedConsole.clear = consoleObj.clear;
            modifiedConsole = true;
        }
        if (consoleObj.dir === undefined || (consoleObj.dir && consoleObj.dir.toString && consoleObj.dir.toString() === "\nfunction dir() {\n    [native code]\n}\n")) {
            consoleObj.dir = function (obj) {
                remoteCode.onConsoleFunc("dir", obj);
            };
            injectedConsole.dir = consoleObj.dir;
            modifiedConsole = true;
        }
        
        // The following functions are not defined on the default console object, so define them now
        if (consoleObj.cd === undefined) {
            consoleObj.cd = function (obj) {
                remoteCode.onConsoleFunc("cd", { obj: obj, argsCount: arguments.length });
            };
            injectedConsole.cd = consoleObj.cd;
            modifiedConsole = true;
        }
        if (consoleObj.select === undefined) {
            consoleObj.select = function (obj) {
                remoteCode.onConsoleFunc("select", { obj: obj });
            };
            injectedConsole.select = consoleObj.select;
            modifiedConsole = true;
        }

        // Add the break mode function
        realWindow.__VISUALSTUDIO_CONSOLE_BREAKMODE_FUNC = function (id, data) {
            remoteCode.onBreakModeFunc(id, data);
        };

        if (modifiedConsole) {
            // Add the detach handler to remove the console object when we detach
            var detachHandler = function () {
                remoteCode.onDetach(realWindow, injectedConsole);
                toolUI.removeEventListener("detach", detachHandler);
            };

            // Add the unload handler to remove the detach handler on a navigate,
            // The actual page navigation will take care of cleaning up the consoles.
            // This enables us to still catch any console function calls during the unload event.
            if (realWindow.addEventListener) {
                // DocumentMode 9 and above
                realWindow.addEventListener("unload", function () {
                    toolUI.removeEventListener("detach", detachHandler);
                });
            } else {
                // Below DocumentMode 9
                realWindow.attachEvent("onunload", function () {
                    toolUI.removeEventListener("detach", detachHandler);
                });
            }

            toolUI.addEventListener("detach", detachHandler);
        }
    },

    onBeforeScriptExecute: function console$remoteCode$onBeforeScriptExecute(dispatchWindow) {
        /// <summary>
        ///     This method is called back when the main browser is about to execute script, 
        ///     so we should insert the console object (and set up handlers to remove it on detach
        /// </summary>   
        /// <param name="dispatchWindow" type="Object">
        ///     The IDispatch window that triggered the BeforeScriptExecute event
        /// </param>

        var realWindow;
        try {
            // Try to get the window object that javascript expects
            realWindow = dispatchWindow.document.parentWindow;
        } catch (ex) {
            // Ignore this beforeScriptExecute, as the window is not valid and cannot attach a console
            return;
        }

        // Ensure the new window is the top level one and not a sub frame
        if (realWindow === mainBrowser.document.parentWindow) {
            // Set this as the current context
            remoteCode.currentWindowContext = realWindow;

            // Context has changed so we need to re-evaluate the constructors
            remoteCode.constructors = null;

            // Notify the console that we have navigated
            if (remoteHelpers.port) {
                // Finish posting any messages from the previous page
                remoteHelpers.postAllMessages();

                remoteCode.initializePage();
            }
        } else {
            // Ensure we re-evaluate the constructors since a frame has changed
            remoteCode.constructors = null;
            
            try {
                // Try to access the current cd window
                var currentWindow = remoteCode.currentWindowContext.document.parentWindow;
            } catch (e) {
                // If we get an exception, it means the currently active cd window was replaced by this new one
                remoteCode.currentWindowContext = realWindow;
            }

            // Sub frames should get a new console
            remoteCode.initializeConsoles(realWindow);
        }

    },

    onDetach: function console$remoteCode$onDetach(realWindow, injectedConsole) {
        /// <summary>
        ///     Cleans up any injected console functions by removing them
        /// </summary>
        /// <param name="realWindow" type="Object">
        ///     The window object that we add the console to
        /// </param>
        /// <param name="injectedConsole" type="Object">
        ///     The object that represents the functions that were injected
        /// </param>        
        /// <disable>JS2078.DoNotDeleteObjectProperties</disable>

        // Reset the document timeout
        remoteHelpers.initializeDocumentTries = 0;

        try {
            if (realWindow.console) {
                // Remove all injected console functions if they haven't been modified
                var functionsNotRemoved = 0;
                for (var prop in injectedConsole) {
                    if (realWindow.console[prop] === injectedConsole[prop]) {
                        if (realWindow.document.documentMode < 9) {
                            // In IE8 mode and below, the delete operator causes the property to go into an 'unknown' state
                            realWindow.console[prop] = null;
                        } else {
                            delete realWindow.console[prop];
                        }
                    } else {
                        functionsNotRemoved++;
                    }
                }

                // If we removed all the injected functions, we might need to delete the console
                if (functionsNotRemoved === 0) {

                    // Check if we should delete the console as well
                    var deleteConsole = true;
                    for (var i in realWindow.console) {
                        if (realWindow.console[i]) {
                            deleteConsole = false;
                            break;
                        }
                    }

                    if (deleteConsole) {
                        delete realWindow.console;
                    }
                }
            }

            if (realWindow.__VISUALSTUDIO_CONSOLE_BREAKMODE_FUNC) {
                // Remove the break mode function
                delete realWindow.__VISUALSTUDIO_CONSOLE_BREAKMODE_FUNC;
            }

            if (realWindow.__VISUALSTUDIO_CONSOLE_INVOKER) {
                // Remove the invoker function
                delete realWindow.__VISUALSTUDIO_CONSOLE_INVOKER;
            }
            if (realWindow.__VISUALSTUDIO_CONSOLE_SELECTED_ELEMENT) {
                // Remove the stored select element
                delete mainBrowser.document.parentWindow.__VISUALSTUDIO_CONSOLE_SELECTED_ELEMENT;
            }
        }
        catch (ex) {
            // We should fail gracefully if there are access issues with already removed consoles
        }

        injectedConsole = null;
    },

    createInvoker: function console$remoteCode$createInvoker(win) {
        /// <summary>
        ///     Creates the visual studio console invoker function on the specified window
        /// </summary>   
        /// <param name="win" type="Object">
        ///     The window onto which to attach the function
        /// </param>

        // This invoker allows us to catch errors in the page's script engine and avoid invoking the debugger.
        // Note that "execScript" is used so that the script is executed in the global context
        // execScript does not return a result, so parameters are passed via properties on our own global function.
        // Using window.eval.call(window, script) would require IE9+.
        
        // Warn the user if the eval function has been modified
        var evalString = win.eval.toString();
        if (evalString !== "\nfunction eval() {\n    [native code]\n}\n" && remoteCode.notifyCallback) {
            remoteCode.notifyCallback({inputId: -1, notifyType: consoleUtils.consoleNotifyType.error, localizeId: "ModifiedEvalFunction"});
            // Identify that the eval function has been altered, so we can report it in any 'NFEs' that may occur
            remoteHelpers.isEvalModified = true;
        }
        
        // Warn the user if the execScript function has been modified
        var execScriptString = win.execScript.toString();
        if (execScriptString !== "\nfunction execScript() {\n    [native code]\n}\n" && remoteCode.notifyCallback) {
            remoteCode.notifyCallback({inputId: -1, notifyType: consoleUtils.consoleNotifyType.error, localizeId: "ModifiedExecScriptFunction"});
            // Identify that the execScript function has been altered, so we can report it in any 'NFEs' that may occur
            remoteHelpers.isExecScriptModified = true;
        }
        
        win.eval(
            "function __VISUALSTUDIO_CONSOLE_INVOKER(input) {\n" +
            "    // This function is used to evaluate commands for the Visual Studio JavaScript Console\n" +
            "    try {\n" +
            "       var inlineConsole = {cd: console.cd, dir: console.dir, select: console.select};\n" +
            "\n" +
            "       if ((typeof $) === 'undefined') {\n" +
            "          inlineConsole.$ = function () { return document.getElementById.apply(document, arguments); };\n" +
            "       }\n" +
            "\n" +
            "       if ((typeof $$) === 'undefined') {\n" +
            "          inlineConsole.$$ = function () { return document.querySelectorAll.apply(document, arguments); };\n" +
            "       }\n" +
            "\n" +
            "       if ((typeof __VISUALSTUDIO_DOMEXPLORER_STORED_ELEMENTS) !== 'undefined') {\n" +
            "          if ((typeof $0) === 'undefined') {\n" +
            "             inlineConsole.$0 = __VISUALSTUDIO_DOMEXPLORER_STORED_ELEMENTS[0];\n" +
            "          }\n" +
            "          if ((typeof $1) === 'undefined') {\n" +
            "             inlineConsole.$1 = __VISUALSTUDIO_DOMEXPLORER_STORED_ELEMENTS[1];\n" +
            "          }\n" +
            "          if ((typeof $2) === 'undefined') {\n" +
            "             inlineConsole.$2 = __VISUALSTUDIO_DOMEXPLORER_STORED_ELEMENTS[2];\n" +
            "          }\n" +
            "          if ((typeof $3) === 'undefined') {\n" +
            "             inlineConsole.$3 = __VISUALSTUDIO_DOMEXPLORER_STORED_ELEMENTS[3];\n" +
            "          }\n" +
            "          if ((typeof $4) === 'undefined') {\n" +
            "             inlineConsole.$4 = __VISUALSTUDIO_DOMEXPLORER_STORED_ELEMENTS[4];\n" +
            "          }\n" +
            "       }\n" +
            "\n" +
            "       __VISUALSTUDIO_CONSOLE_INVOKER.inlineConsole = inlineConsole;\n" +
            "       __VISUALSTUDIO_CONSOLE_INVOKER.input = input;\n" +
            "       __VISUALSTUDIO_CONSOLE_INVOKER.returnValue = { isError:false };\n" +
            "\n" +
            "       execScript('/* VS JSConsole invoke */ " +
            "          try {" +
            "              with (__VISUALSTUDIO_CONSOLE_INVOKER.inlineConsole) {" +
            "                  __VISUALSTUDIO_CONSOLE_INVOKER.returnValue.result = eval(__VISUALSTUDIO_CONSOLE_INVOKER.input);" +
            "              }" +
            "          } catch(e) {" +
            "              __VISUALSTUDIO_CONSOLE_INVOKER.returnValue.result = (e.message || e.description);" +
            "              __VISUALSTUDIO_CONSOLE_INVOKER.returnValue.isError = true;" +
            "          }" +
            "       ');\n" +
            "\n" +
            "       var returnValue = __VISUALSTUDIO_CONSOLE_INVOKER.returnValue;\n" +
            "\n" +
            "       __VISUALSTUDIO_CONSOLE_INVOKER.inlineConsole = null;\n" +
            "       __VISUALSTUDIO_CONSOLE_INVOKER.input = null;\n" +
            "       __VISUALSTUDIO_CONSOLE_INVOKER.returnValue = null;\n" +
            "\n" +
            "       return returnValue;\n" +
            "   } catch(invokerEx) {\n" +
            "       /* This exception occurs if the evaluation thread was terminated during the call due to a refresh, so fail gracefully. */ \n" +
            "       return { result: undefined, isError: false };\n" +
            "   }\n" +
            "}"
        );
    },

    onConsoleMessage: function console$remoteCode$onConsoleMessage(source, level, messageId, messageText, fileUrl, lineNumber, columnNumber) {
        /// <summary>
        ///     This method is called from code within IE that wants to output to the console
        /// </summary>   
        /// <param name="source" type="String">
        ///     The source identifier that created the message
        /// </param>   
        /// <param name="level" type="String">
        ///     The level of the message
        /// </param>   
        /// <param name="messageId" type="Number">
        ///     The message id for the type of message coming from the console
        /// </param>   
        /// <param name="messageText" type="String">
        ///     The text to display
        /// </param>   
        /// <param name="fileUrl" type="String">
        ///     The file url that caused the message
        /// </param>   
        /// <param name="lineNumber" type="Number">
        ///     The line number for the message
        /// </param>   
        /// <param name="columnNumber" type="Number">
        ///     The column number for the message
        /// </param>   

        if (source === "CONSOLE") {
            switch (messageId) {
                case 6000: // Log
                    remoteCode.onConsoleFunc(consoleUtils.consoleNotifyType.log, messageText);
                    return;
                case 6001: // Warn
                    remoteCode.onConsoleFunc(consoleUtils.consoleNotifyType.warn, messageText);
                    return;
                case 6002: // Error
                    remoteCode.onConsoleFunc(consoleUtils.consoleNotifyType.error, messageText);
                    return;
                case 6003: // Assert
                    remoteCode.onConsoleFunc(consoleUtils.consoleNotifyType.assert, messageText);
                    return;
                case 6004: // Info
                    remoteCode.onConsoleFunc(consoleUtils.consoleNotifyType.info, messageText);
                    return;
            }
        }

        var messageIdentifier = source + messageId;
        var data = { messageId: messageIdentifier, message: messageIdentifier + ": " + messageText, fileUrl: fileUrl, lineNumber: lineNumber, columnNumber: columnNumber };

        switch (level) {
            case 0:
                remoteCode.onConsoleFunc(consoleUtils.consoleNotifyType.info, data);
                break;
            case 1:
                remoteCode.onConsoleFunc(consoleUtils.consoleNotifyType.warn, data);
                break;
            case 2:
                remoteCode.onConsoleFunc(consoleUtils.consoleNotifyType.error, data);
                break;
        }

    },

    onConsoleFunc: function console$remoteCode$onConsoleFunc(functionId, data) {
        /// <summary>
        ///     This method is called by functions attached to the window.console
        ///     It processes the function call and communicates information back to the VS side
        /// </summary>
        /// <param name="functionId" type="String">
        ///     The type of the message
        ///     Either a notification type from consoleUtils.consoleNotifyType or a built in function
        ///     such as "cd", "clear", "dir", etc
        /// </param>
        /// <param name="data" type="Object" optional="true">
        ///     The information used for this function call
        /// </param>


        switch (functionId) {
            case "cd":
                // Built in console.cd command
                if (remoteCode.notifyCallback) {
                    try {
                        var iframe;
                        if (data.argsCount === 0) {
                            // Set the context back to the main window
                            iframe = mainBrowser.document.parentWindow;
                        } else {
                            // CD into the window
                            iframe = domHelper.getCrossSiteWindow(remoteCode.currentWindowContext, data.obj);
                        }

                        remoteCode.currentWindowContext = iframe;
                        remoteCode.constructors = null;
                        
                        var newWindowContext = iframe.eval("window.location.href");
                        var newWindowText = iframe.eval("window.location.hostname") + iframe.eval("window.location.pathname");

                        // Remove trailing slash
                        newWindowText = newWindowText.replace(/\/$/, "");
                        
                        remoteCode.notifyCallback({ notifyType: "consoleItemCDContext", message: newWindowText, contextInfo: newWindowContext });

                    } catch (e) {
                        remoteCode.notifyCallback({ notifyType: consoleUtils.consoleNotifyType.error, message: (e.message || e.description) });
                    }
                }
                break;

            case "clear":
                // Built in console.clear command
                if (remoteCode.clearCallback) {
                    remoteCode.reset();
                    remoteCode.clearCallback();
                }
                break;

            case "dir":
                // Built in console.dir command
                if (remoteCode.outputCallback) {
                    var returnObj = remoteCode.createOutputObject(-1, data);

                    if (returnObj.detailedType !== "undefined") {
                        remoteCode.outputCallback(returnObj);
                    }
                }
                break;

            case "select":
                // Built in console.select command
                if (remoteCode.notifyCallback) {
                    remoteCode.currentWindowContext.__VISUALSTUDIO_CONSOLE_SELECTED_ELEMENT = data.obj;

                    // Check that this is actually an html element
                    remoteCode.ensureConstructorsAreAvailable(true);
                    var isHtmlElement = (data.obj instanceof remoteCode.constructors.htmlElement);

                    remoteCode.notifyCallback({ notifyType: "consoleItemSelectInDom", message: isHtmlElement });
                }
                break;

            default:
                // All other functions are notification types
                if (!remoteCode.notifyCallback) {
                    // The port is not ready yet, so queue up the notifications
                    remoteCode.consoleNotificationQueue.push({ functionId: functionId, data: data });
                } else {
                    // Fire the notification directly

                    remoteCode.notifyCallback({ notifyType: functionId, message: data });
                }
                break;
        }
    },

    onBreakModeFunc: function console$remoteCode$onBreakModeFunc(id, data) {
        /// <summary>
        ///     This method is called when the console is at a break point, and is
        ///     used to evaluate commands while in that state
        /// </summary>
        /// <param name="id" type="String">
        ///     The id associated with the command
        ///     This is in the form:
        ///     "functionName:id:callbackUid"
        /// </param>
        /// <param name="data" type="Object">
        ///     The information used for this function call
        /// </param>

        // Since we are being executed in break mode, we don't know the current 'eval' context, as
        // it is handled by VS. So we need to re-evaluate the constructors in this context.
        remoteCode.constructors = null;

        if (id === "") {
            // This is a regular CallProxy call, so just process the messages
            remoteHelpers.processMessages({ data: data });
        } else {

            // First get the info about this call from the generic id
            var parts = id.split(":");
            if (parts.length === 3) {
                var funcName = parts[0];
                var inputId = parseInt(parts[1], 10);
                var uid = parts[2];

                // Now perform the requested function
                var returnValue;
                if (funcName === "processInput") {

                    // The constructors will have been created by the VS debug call so as to
                    // use the correct context.
                    if (data.constructors) {
                        remoteCode.constructors = data.constructors;
                    }

                    // Get the console result
                    returnValue = remoteCode.createConsoleResult(inputId, data);
                }

                // Now return the result using the callback mechanism
                if (returnValue !== undefined) {
                    remoteHelpers.postObject({
                        uid: uid,
                        args: [returnValue]
                    });
                }
            }
        }

        // We need to reset the constructors again, in case we switch back to run mode under a difference context
        remoteCode.constructors = null;
    },

    ensureConstructorsAreAvailable: function console$remoteCode$ensureConstructorsAreAvailable(forceCreate) {
        /// <summary>
        ///     Gets Html type name for a specified object
        /// </summary>
        /// <param name="forceCreate" type="Boolean" optional="true">
        ///     Optional parameter that specifies if we should re-create the constructors if they already exist
        /// </param>

        if (!remoteCode.constructors || forceCreate) {
            // For detailedType checks we need to evaluate some code in the context of the browser to
            // get the constructors for our metadata types.
            remoteCode.constructors = remoteCode.currentWindowContext.eval(
                "(function (){\n" +
                "   var constructors = {};\n" +
                "   try {\n" +
                "      constructors.array = (new Array).constructor;\n" +
                "      constructors.date = (new Date).constructor;\n" +
                "      constructors.regex = (new RegExp).constructor;\n" +
                "      constructors.htmlElement = window.HTMLElement;\n" +
                "      constructors.htmlNode = window.Node;\n" +
                "      constructors.nodeList = window.NodeList;\n" +
                "      constructors.htmlCollection = window.HTMLCollection;\n" +
                "   } catch (e) {\n" +
                "   }\n" +
                "   return constructors;\n" +
                " })();"
            );
        }

    },

    getHtmlViewableTypeName: function console$remoteCode$getHtmlViewableTypeName(obj) {
        /// <summary>
        ///     Gets Html type name for a specified object
        /// </summary>
        /// <param name="obj" type="Object">
        ///     The javascript object that we need the type name for
        /// </param>
        /// <returns type="String">
        ///     The html type name of the object, or null if the object is not an Html element
        /// </returns>

        // Create the constructors
        remoteCode.ensureConstructorsAreAvailable();

        if (remoteCode.constructors) {
            if (remoteCode.constructors.htmlElement && (obj instanceof remoteCode.constructors.htmlElement)) {
                // HTMLElement
                return "HtmlElement";
            } else if (remoteCode.constructors.htmlNode && (obj instanceof remoteCode.constructors.htmlNode)) {
                    // Now that we know this is an html node, we need to check its nodeType
                if (obj.nodeType === obj.DOCUMENT_NODE) {
                    // Document
                    return "DocumentNode";
                } else if (obj.nodeType === obj.ATTRIBUTE_NODE) {
                    // Attribute
                    return "AttributeNode";
                } else {
                    // HtmlNode
                    return "HtmlNode";
                }
            } else if (remoteCode.constructors.nodeList && (obj instanceof remoteCode.constructors.nodeList)) {
                    // NodeList
                return "NodeList";
            } else if (remoteCode.constructors.htmlCollection && (obj instanceof remoteCode.constructors.htmlCollection)) {
                    // HTMLCollection
                return "HtmlCollection";
            }
        }

        // Object 
        return null;
    },

    createConsoleResult: function console$remoteCode$createConsoleResult(inputId, evaluatedReturnValue) {
        /// <summary>
        ///     Creates the correct representation of an VisualStudioInvoker result
        /// </summary>
        /// <param name="inputId" type="String">
        ///     The identifier of the input command that was evaluated to create this object
        /// </param>
        /// <param name="evaluatedReturnValue" type="Object">
        ///     The result returned from the VisualStudioConsoleInvoker
        /// </param>
        /// <returns type="Object">
        ///     The constructed javscript object that is ready to be stringified and sent over to the VS side
        /// </returns>
        /// <disable>JS2021.SeparateBinaryOperatorArgumentsWithSpaces</disable>

        // Create the constructors
        remoteCode.ensureConstructorsAreAvailable();

        if (evaluatedReturnValue.isError) {
            // Error values should be shown using a notification
            if (remoteCode.notifyCallback) {
                remoteCode.notifyCallback({ inputId: inputId, notifyType: consoleUtils.consoleNotifyType.error, message: evaluatedReturnValue.result });
            }
        } else {

            // Mark the start of the object creation
            mainBrowser.document.parentWindow.msWriteProfilerMark("ConsoleRemote:BeginCreateResultObject");

            var consoleObject;

            // Now check what type of item we need to return
            var htmlTypeName = remoteCode.getHtmlViewableTypeName(evaluatedReturnValue.result);

            // Check for an html viewable type, excluding documents which should be objects by default (but have a view as html option)
            if (htmlTypeName !== null && htmlTypeName !== "DocumentNode") {
                // Create an Html Viewable result
                consoleObject = remoteCode.createOutputHtmlElement(inputId, evaluatedReturnValue.result, htmlTypeName);
            } else {
                // Create an Object result
                consoleObject = remoteCode.createOutputObject(inputId, evaluatedReturnValue.result);

                // Document nodes should be marked as 'Html Viewable'
                if (htmlTypeName === "DocumentNode") {
                    consoleObject.isHtmlViewableType = true;
                }
            }

            // Mark the end of the object creation
            mainBrowser.document.parentWindow.msWriteProfilerMark("ConsoleRemote:EndCreateResultObject");

            // Return the created object
            return consoleObject;
        }
    },

    createOutputHtmlElement: function console$remoteCode$createOutputHtmlElement(inputId, element, htmlTypeName) {
        /// <summary>
        ///     Creates a representation of a javascript HtmlElement for the VS console
        /// </summary>
        /// <param name="inputId" type="String">
        ///     The identifier of the input command that was evaluated to create this object
        /// </param>
        /// <param name="element" type="Object">
        ///     The object to traverse and convert to a console object
        ///     This should be from the result of an eval
        /// </param> 
        /// <param name="htmlTypeName" type="String" >
        ///     Specifies the html type Name
        /// </param>          
        /// <returns type="Object">
        ///     The constructed javscript object representing the HtmlElement
        /// </returns>

        var name;
        try {
            name = Object.prototype.toString.call(element);
        } catch (e) {
            // A failure means that this is a cross-site window that we cannot access
            name = null;
        }

        // Create the html node, and remember we want to see empty text elements
        var value = htmlTreeHelpers.createMappedNode(element, true);

        if (htmlTypeName === "NodeList" ||
            htmlTypeName === "HtmlCollection") {
            // This is a list so store that info
            value.tag = htmlTypeName;
            htmlTreeHelpers.mapping[value.uid].listType = htmlTypeName;
            value.attributes = [{ name: "length", value: element.length }];
        }

        // Return this console object
        return {
            inputId: inputId,
            consoleType: "consoleItemOutput",
            detailedType: "htmlElement",
            isExpandable: true,
            isHtmlViewableType: true,
            name: name,
            value: value,
            uid: 0
        };
    },

    createOutputObject: function console$remoteCode$createOutputObject(inputId, obj) {
        /// <summary>
        ///     Creates a representation of a javascript object for the VS console
        /// </summary>
        /// <param name="inputId" type="String">
        ///     The identifier of the input command that was evaluated to create this object
        /// </param>
        /// <param name="obj" type="Object">
        ///     The object to traverse and convert to a console object
        ///     This should be from the result of an eval
        /// </param>
        /// <returns type="Object">
        ///     The constructed javscript object representing the object
        /// </returns>

        // The meta type of the evaluate result could be 'string', 'object', 'array', 'date', 'regex', etc
        var detailedType = remoteCode.getDetailedTypeOf58(obj, remoteCode.constructors);

        // Expandable objects are not empty because they have properties
        var isExpandable = !remoteCode.isEmpty58(obj);

        // Get the value for the object
        var name;
        var value;
        var uid;
        if (isExpandable) {
            name = remoteCode.createName(obj, detailedType);

            // We need to get the special window object
            if (name === "[object Window]") {
                try {
                    var iframe = domHelper.getCrossSiteWindow(remoteCode.currentWindowContext, obj);
                    if (iframe) {
                        obj = iframe;
                    }
                } catch (e) {
                    // We couldn't access this window, so create a fake value
                    value = { 0: remoteCode.createExceptionValue(e) };
                }
            }

            if (!value) {
                uid = remoteHelpers.getUid();
                value = remoteCode.createValue(obj, uid);
                remoteCode.resultMap[uid] = obj;
            }
        } else {
            name = null;
            value = remoteCode.createValue(obj);
        }

        var htmlTypeName = remoteCode.getHtmlViewableTypeName(obj);
        var isHtmlViewableType = (htmlTypeName !== null && htmlTypeName !== "AttributeNode");

        // Return this console object
        return {
            inputId: inputId,
            consoleType: "consoleItemOutput",
            detailedType: detailedType,
            isExpandable: isExpandable,
            isHtmlViewableType: isHtmlViewableType,
            name: name,
            value: value,
            uid: uid
        };
    },

    runInPage: function console$remoteCode$runInPage(functionToRun, arg) {
        /// <summary>
        ///     Runs functionToRun in the context of the page passing 
        ///     in arg as the only argument by stringifying it then 
        ///     using the main window to eval it.
        /// </summary>
        /// <param name="functionToRun" type="Function">
        ///     A function containing code to run in the page.
        /// </param>
        /// <param name="arg" type="Object">
        ///     An argument to pass to functionToRun
        /// </param>
        /// <returns type="Object">
        ///     The result of running the function with the specified
        ///     argument.
        /// </returns>

        // Make sure there is an invoker to evaluate against
        if (!remoteCode.currentWindowContext.__VISUALSTUDIO_CONSOLE_INVOKER) {
            remoteCode.createInvoker(remoteCode.currentWindowContext);
        }
        var invoker = mainBrowser.document.parentWindow.__VISUALSTUDIO_CONSOLE_INVOKER;
        var temp = invoker.objParam;
        invoker.objParam = arg;

        var result = mainBrowser.document.parentWindow.eval(
            "(function() { try { return (" + functionToRun.toString() + ")(__VISUALSTUDIO_CONSOLE_INVOKER.objParam); } catch (e) { return 'undefined'; } })();");

        invoker.objParam = temp;
        return result;
    },

    getTypeOf: function console$remoteCode$getTypeOf(obj) {
        /// <summary>
        ///     Because the 5.8 engine can't get accurate typeof info for
        ///     Chakra objects, if we're not running under Chakra, we must
        ///     go to the page for typeofs.
        /// </summary>
        /// <param name="obj" type="Object">
        ///     The javascript value to be checked for its type
        /// </param>
        /// <returns type="String">
        ///     A string representing the type of this value, possible values include:
        ///     undefined, null, object, string, number, function, and boolean
        /// </returns>

        if (toolUI.engineSupportsES5) {
            return typeof obj;
        } else {
            return remoteCode.runInPage(function (arg) { return typeof arg; }, obj);
        }
    },

    getDetailedTypeOf58: function console$remoteCode$getDetailedTypeOf58(value, constructors) {
        /// <summary>
        ///     Gets a string representing the type of the passed in value.
        ///     This supliments the built in typeof function by calculating the type of certain objects such as
        ///     array, date, and regex.  The 58 version of this function is designed to work for the jscript 5.8 engine
        ///     talking to a page running jscript9
        /// </summary>
        /// <param name="value" type="Object">
        ///     The javascript value to be checked for its real type
        /// </param>
        /// <param name="constructors" type="Object" optional="true">
        ///     An optional object that contains a set of constructors to check an "object" type against for further
        ///     sub typing into array, date, and regex
        ///     These objects need to come from the same window context as the value we are classifying
        /// </param>
        /// <returns type="String">
        ///     A string representing the type of this value, possible values include:
        ///     undefined, null, object, array, date, regex, string, number, function, and boolean
        /// </returns>
        /// <disable>JS3053.IncorrectNumberOfArguments,JS2005.UseShortFormInitializations</disable>
        if (value === undefined) {
            return "undefined";
        }

        var type = remoteCode.getTypeOf(value);

        if (type === "object") {
            if (value) {
                try {
                    if (typeof value.length === "number" &&
                        (toolUI.engineSupportsES5 ? typeof value.propertyIsEnumerable :
                        remoteCode.getTypeOf(value.propertyIsEnumerable) === "function" &&
                        !(value.propertyIsEnumerable("length")) &&
                        remoteCode.getTypeOf(value.splice) === "function")) {
                        return "array";
                    }

                    // See if we have specific constructors to test against
                    var arrayCon = (constructors && constructors.array ? constructors.array : remoteCode.runInPage(function () { return (new Array()).constructor; }, null));
                    var dateCon = (constructors && constructors.date ? constructors.date : remoteCode.runInPage(function () { return (new Date()).constructor; }, null));
                    var regexCon = (constructors && constructors.regex ? constructors.regex : remoteCode.runInPage(function () { return (new RegExp()).constructor; }, null));

                    if (value.constructor === arrayCon) {
                        return "array";
                    } else if (value.constructor === dateCon) {
                        return "date";
                    } else if (value.constructor === regexCon) {
                        return "regex";
                    }
                } catch (e) {
                    // This object is not accessible
                }
            } else {
                return "null";
            }

            return "object";
        }

        return type;
    },

    isEmpty58: function console$remoteCode$isEmpty58(obj) {
        /// <summary>
        ///     Checks if the passed in object is empty (such as { }) because it has no members
        /// </summary>
        /// <param name="obj" type="Object">
        ///     The javascript object to be checked for members
        /// </param>
        /// <returns type="Boolean">
        ///     True if the object contains no members, False otherwise
        /// </returns>

        var pageTypeOf = remoteCode.getTypeOf(obj);
        if (pageTypeOf === "object" || pageTypeOf === "function") {
            try {
                for (var i in obj) {
                    return false;
                }
            } catch (e) {
                // Cannot access this member therefore it must have one
                return false;
            }
        }

        return true;
    },

    getPrototypeName: function console$remoteCode$getPrototypeName(obj) {
        /// <summary>
        ///     Grabs the toString of an object's prototype, using the page's engine if 
        ///     we're running under 5.8.  This is for use in identifying the type 
        ///     of object when outputting to the console.
        /// </summary>
        /// <param name="obj" type="Object">
        ///     The object whose prototype name we are fetching
        /// </param>
        /// <returns type="String">
        ///     toString of the object's prototype.
        /// </returns>
        var prototypeName;
        if (toolUI.engineSupportsES5) {
            try {
                prototypeName = Object.prototype.toString.call(obj);
            } catch (ex) {
                // A failure means that this is a cross-site window that we cannot access
                prototypeName = null;
            }
        } else {
            // Try to get the type name
            prototypeName = remoteCode.runInPage(
            function (param) {
                var retVal;
                try {
                    retVal = Object.prototype.toString.call(param);
                } catch (ex) {
                    // A failure means that this is a cross-site window that we cannot access
                    retVal = null;
                }
                return retVal;
            },
            obj);
        }
        return prototypeName;
    },

    createName: function console$remoteCode$createName(obj, detailedType) {
        /// <summary>
        ///     Creates a string representation of a javascript object's contents
        /// </summary>
        /// <param name="obj" type="Object">
        ///     The object to traverse and convert to a console object
        ///     This should be from the result of an eval
        /// </param>
        /// <param name="detailedType" type="String" optional="true">
        ///     If we already know the detailed type of obj, pass it in to 
        ///     avoid recalculating it.
        /// </param>
        /// <returns type="String">
        ///     The constructed string representing the object
        /// </returns>
        if (!detailedType) {
            detailedType = remoteCode.getDetailedTypeOf58(obj, remoteCode.constructors);
        }

        switch (detailedType) {
            case "boolean":
                return "" +  obj;
            case "date":
                return "[date] " + obj;

            case "function":
                return "" + obj;

            case "null":
                return "null";

            case "number":
                return  "" + obj;

            case "regex":
                return "[regex] " + obj;

            case "string":
                return "\"" + obj + "\"";

            case "undefined":
                return "undefined";

            case "array":
                if (remoteCode.isEmpty58(obj)) {
                    return consoleUtils.consoleUITypeStrings.emptyArray;
                }
                return remoteCode.getPrototypeName(obj);

                // FALLTHROUGH
            case "object":
            case "htmlElement":
            if (remoteCode.isEmpty58(obj)) {
                    return consoleUtils.consoleUITypeStrings.emptyObject;
                }
                return remoteCode.getPrototypeName(obj);
            default:
                return "" + obj;
        }
    },


    createValue: function console$remoteCode$createValue(obj, parentUid) {
        /// <summary>
        ///     Creates a representation of a value for the VS console
        /// </summary>
        /// <param name="obj" type="Object">
        ///     The object to traverse and create the value for
        /// </param>
        /// <param name="parentUid" type="String" optional="true">
        ///     The indentifier of the object in the stored map
        ///     This is used later to expand child elements
        /// </param>
        /// <returns type="Object">
        ///     The constructed javscript object representing the value
        /// </returns>

        var detailedType = remoteCode.getDetailedTypeOf58(obj, remoteCode.constructors);
        var isEmpty = remoteCode.isEmpty58(obj);
        
        // We can shortcut empty objects and just return their string value
        if (isEmpty) {
            return remoteCode.createName(obj, detailedType);
        }

        // Loop through the children and generate the meta object for it
        var outputObj = [];
        
        try {
            for (var i in obj) {
                // Some children cannot be accessed (eg document.fileUpdatedDate)
                // So first perform a check that we can use this child.
                var child;
                try {
                    child = obj[i];
                
                    // Get info about this child node
                    var childdetailedType = remoteCode.getDetailedTypeOf58(child, remoteCode.constructors);

                    var childIsExpandable = !remoteCode.isEmpty58(child);

                    // Try to get the type name, this will fail on cross site windows
                    var childName = null;
                    if (childIsExpandable) {
                        childName = remoteCode.createName(obj[i], childdetailedType);
                    }

                    var htmlTypeName = remoteCode.getHtmlViewableTypeName(child);
                    var isHtmlViewableType = (htmlTypeName !== null && htmlTypeName !== "AttributeNode");

                    // Create the child object
                    outputObj.push({
                        propertyName: i, 
                        propertyValue: {
                            detailedType: childdetailedType,
                            isExpandable: childIsExpandable,
                            isHtmlViewableType: isHtmlViewableType,
                            name: childName,
                            value: (!childIsExpandable ? remoteCode.createValue(child) : parentUid + ":" + i)
                        }
                    });
                } 
                catch (childAccessEx) {
                    // Child inaccessible, so just just return a fake value
                    outputObj.push({propertyName: i, propertyValue: remoteCode.createExceptionValue(childAccessEx)});
                    continue;
                }
            }
        } catch (objectAccessEx) {
            // The object was inaccessible, (it was probably a window object), so just return a fake value
            outputObj.push({propertyName: "0", propertyValue: remoteCode.createExceptionValue(objectAccessEx)});
        }
        return outputObj;
    },

    createExceptionValue: function console$remoteCode$createExceptionValue(exception) {
        /// <summary>
        ///     Creates a representation of an exception for the VS console
        /// </summary>
        /// <param name="exception" type="Object">
        ///     The exception to create the object for
        /// </param>
        /// <returns type="Object">
        ///     The constructed javscript object representing the exception
        /// </returns>

        var information = (exception.message || exception.description);
        var msg = "<" + (information).replace(/^\s+|\s+$/g, "") + ">";
        if (msg === "<>") {
            // This string matches the exception from javascript and should not be localized
            msg = "<Access denied.>";
        }

        return {
            detailedType: "exception",
            isExpandable: false,
            value: msg
        };
    },

    reset: function () {
        /// <summary>
        ///     Reset settings back to their original values
        /// </summary>

        remoteHelpers.uid = 0;
        remoteCode.resultMap = {};
        htmlTreeHelpers.reset();
    }
};

messageHandlers = {
    /// <summary>
    ///     Object that acts as the message handler for messages that get sent from the VS side
    ///     The messages contain a command property that corrisponds to a function on this 
    ///     object that processes that message
    /// </summary>

    getDocumentMode: function console$messageHandlers$getDocumentMode() {
        /// <summary>
        ///     Gets the document mode of the main browser page
        /// </summary>
        /// <returns type="Number">
        ///     The document mode of the main browser page
        /// </returns> 

        return mainBrowser.document.documentMode;
    },

    clearConsoleData: function console$messageHandlers$clearConsoleData() {
        /// <summary>
        ///     Removes any stored data for the remote side of the console
        ///     This is triggered when the user clicks the clear button on the VS side
        /// </summary>

        remoteCode.reset();
    },

    registerConsoleCallbacks: function console$messageHandlers$registerConsoleCallbacks(outputCallback, notifyCallback, clearCallback) {
        /// <summary>
        ///     Stores the callbacks used to send messages and commands to the VS side
        /// </summary>
        /// <param name="outputCallback" type="Function">
        ///     The callback function for output results
        /// </param>
        /// <param name="notifyCallback" type="Function">
        ///     The callback function that is used for sending info, warnings, errors, and assert
        ///     to the VS side console
        /// </param>
        /// <param name="clearCallback" type="Function">
        ///     The callback function that will clear the VS side console
        /// </param>        

        // Store the callback for console output messages
        remoteCode.outputCallback = outputCallback;

        // Store the callback for console notification messages
        remoteCode.notifyCallback = notifyCallback;

        // Store the callback for clearing the console
        remoteCode.clearCallback = clearCallback;

        // If we queued up any messages while the port was being created, then fire them now
        for (var i = 0; i < remoteCode.consoleNotificationQueue.length; i++) {
            var notification = remoteCode.consoleNotificationQueue[i];
            remoteCode.onConsoleFunc(notification.functionId, notification.data);
        }
    },

    processInput: function console$messageHandlers$processInput(inputId, input) {
        /// <summary>
        ///     Evals the input string against the current page under debugging and then
        ///     posts back the result to the VS console
        /// </summary>
        /// <param name="inputId" type="Number">
        ///     The unique id to associate with this script eval, this id will be posted back to the VS side
        ///     along with the result of the eval
        /// </param>
        /// <param name="input" type="String">
        ///     The javascript string that is to be evaled on the page
        /// </param>
        /// <returns type="Object">
        ///     The object to return to VS
        /// </returns>

        mainBrowser.document.parentWindow.msWriteProfilerMark("ConsoleRemote:BeginConsoleInvoke");

        // Make sure there is an invoker to evaluate against
        if (!remoteCode.currentWindowContext.__VISUALSTUDIO_CONSOLE_INVOKER) {
            remoteCode.createInvoker(remoteCode.currentWindowContext);
        }

        // Evaluate the input command
        var returnValue = remoteCode.currentWindowContext.__VISUALSTUDIO_CONSOLE_INVOKER(input);

        mainBrowser.document.parentWindow.msWriteProfilerMark("ConsoleRemote:EndConsoleInvoke");

        return remoteCode.createConsoleResult(inputId, returnValue);
    },

    getObjectChildren: function console$messageHandlers$getObjectChildren(identifier) {
        /// <summary>
        ///     Gets the children for a known object that has already been evaluated in the console
        /// </summary>
        /// <param name="identifier" type="String">
        ///     The unique id that identifies the object we want to expand. 
        ///     This is in the form "uid:property"
        /// </param>
        /// <returns type="Object">
        ///     The object to return to VS, or null if the object wasn't mapped
        /// </returns>

        // Process the identifier
        var parts = identifier.split(":", 2);

        // Ensure that this is a valid id
        if (parts.length === 2) {
            var mappedResult = remoteCode.resultMap[parts[0]];

            // Get the output object
            if (mappedResult) {
                var mappedChild = mappedResult[parts[1]];
                if (mappedChild !== undefined) {
                    return remoteCode.createOutputObject(-1, mappedChild);
                }
            }
        }

        // Specifically return null to indicate that no children were found for this item
        return null;
    },

    getHtmlChildren: function console$messageHandlers$getHtmlChildren(uid) {
        /// <summary>
        ///     Get all the child elements from a particular mapped DOM element
        /// </summary>
        /// <param name="uid" type="String">
        ///     The mapped unique id of the DOM element we want to get the children of
        /// </param>        
        /// <returns type="Array">
        ///     An array of mapped nodes that represent the children of the DOM element
        /// </returns> 

        var mappedNode = htmlTreeHelpers.mapping[uid];
        if (!mappedNode || !htmlTreeHelpers.isElementAccessible(mappedNode.ele)) {
            return;
        }

        return htmlTreeHelpers.getChildrenForMappedNode(uid, true);
    },

    getHtmlAttributes: function console$messageHandlers$getHtmlAttributes(uid) {
        /// <summary>
        ///     Get all the HTML attributes of the current specified DOM Element
        /// </summary>
        /// <param name="uid" type="String">
        ///     The mapped unique id of the DOM element we want to get the attributes for
        /// </param>        
        /// <returns type="Array">
        ///     An array of objects that contain a name and value pair for each attribute
        /// </returns> 

        // Ensure we have already mapped the requested DOM element
        var mappedNode = htmlTreeHelpers.mapping[uid];
        if (!mappedNode || !htmlTreeHelpers.isElementAccessible(mappedNode.ele)) {
            return;
        }

        var allAttributes = [];
        var element = mappedNode.ele;
        if (element.attributes) {
            // Create a name/value pair for each attribute and add it to the array
            for (var i = 0; i < element.attributes.length; i++) {
                allAttributes.push({
                    name: element.attributes[i].name,
                    value: element.attributes[i].value
                });
            }
        }
        return allAttributes;
    },

    getObjectItemAsHtml: function console$messageHandlers$getObjectItemAsHtml(identifier) {
        /// <summary>
        ///     Gets the Html representation of an object that has already been evaluated in the console
        /// </summary>
        /// <param name="identifier" type="String">
        ///     The unique id that identifies the object we want to use
        /// </param>
        /// <returns type="Object">
        ///     The object to return to VS, or null if the object wasn't mapped
        /// </returns>

        var mappedItem = null;

        // Process the identifier
        var parts = identifier.split(":", 2);
        if (parts.length === 2) {
            var mappedParent = remoteCode.resultMap[parts[0]];
            if (mappedParent) {
                mappedItem = mappedParent[parts[1]];
            }
        } else if (parts.length === 1) {
            mappedItem = remoteCode.resultMap[identifier];
        }

        // Get the html object
        if (mappedItem) {
            return remoteCode.createOutputHtmlElement(-1, mappedItem, remoteCode.getHtmlViewableTypeName(mappedItem));
        }


        // Specifically return null to indicate that no object could be found for this item
        return null;
    },

    getHtmlItemAsObject: function console$messageHandlers$getHtmlItemAsObject(uid) {
        /// <summary>
        ///     Gets the object representation of an Html element that has already been evaluated in the console
        /// </summary>
        /// <param name="uid" type="String">
        ///     The unique id that identifies the HtmlElement we want to use
        /// </param>
        /// <returns type="Object">
        ///     The object to return to VS, or null if the object wasn't mapped
        /// </returns>

        // Ensure we have already mapped the requested DOM element
        var mappedNode = htmlTreeHelpers.mapping[uid];
        if (!mappedNode || !htmlTreeHelpers.isElementAccessible(mappedNode.ele)) {
            // Specifically return null to indicate that no object could be found for this item
            return null;
        }

        return remoteCode.createOutputObject(-1, mappedNode.ele);
    }


};

remoteCode.initialize();

// SIG // Begin signature block
// SIG // MIIamwYJKoZIhvcNAQcCoIIajDCCGogCAQExCzAJBgUr
// SIG // DgMCGgUAMGcGCisGAQQBgjcCAQSgWTBXMDIGCisGAQQB
// SIG // gjcCAR4wJAIBAQQQEODJBs441BGiowAQS9NQkAIBAAIB
// SIG // AAIBAAIBAAIBADAhMAkGBSsOAwIaBQAEFCEEMKeTvuIf
// SIG // N4VNZy64NSxUvycXoIIVgjCCBMMwggOroAMCAQICEzMA
// SIG // AAArOTJIwbLJSPMAAAAAACswDQYJKoZIhvcNAQEFBQAw
// SIG // dzELMAkGA1UEBhMCVVMxEzARBgNVBAgTCldhc2hpbmd0
// SIG // b24xEDAOBgNVBAcTB1JlZG1vbmQxHjAcBgNVBAoTFU1p
// SIG // Y3Jvc29mdCBDb3Jwb3JhdGlvbjEhMB8GA1UEAxMYTWlj
// SIG // cm9zb2Z0IFRpbWUtU3RhbXAgUENBMB4XDTEyMDkwNDIx
// SIG // MTIzNFoXDTEzMTIwNDIxMTIzNFowgbMxCzAJBgNVBAYT
// SIG // AlVTMRMwEQYDVQQIEwpXYXNoaW5ndG9uMRAwDgYDVQQH
// SIG // EwdSZWRtb25kMR4wHAYDVQQKExVNaWNyb3NvZnQgQ29y
// SIG // cG9yYXRpb24xDTALBgNVBAsTBE1PUFIxJzAlBgNVBAsT
// SIG // Hm5DaXBoZXIgRFNFIEVTTjpDMEY0LTMwODYtREVGODEl
// SIG // MCMGA1UEAxMcTWljcm9zb2Z0IFRpbWUtU3RhbXAgU2Vy
// SIG // dmljZTCCASIwDQYJKoZIhvcNAQEBBQADggEPADCCAQoC
// SIG // ggEBAKa2MA4DZa5QWoZrhZ9IoR7JwO5eSQeF4HCWfL65
// SIG // X2JfBibTizm7GCKlLpKt2EuIOhqvm4OuyF45jMIyexZ4
// SIG // 7Tc4OvFi+2iCAmjs67tAirH+oSw2YmBwOWBiDvvGGDhv
// SIG // sJLWQA2Apg14izZrhoomFxj/sOtNurspE+ZcSI5wRjYm
// SIG // /jQ1qzTh99rYXOqZfTG3TR9X63zWlQ1mDB4OMhc+LNWA
// SIG // oc7r95iRAtzBX/04gPg5f11kyjdcO1FbXYVfzh4c+zS+
// SIG // X+UoVXBUnLjsfABVRlsomChWTOHxugkZloFIKjDI9zMg
// SIG // bOdpw7PUw07PMB431JhS1KkjRbKuXEFJT7RiaJMCAwEA
// SIG // AaOCAQkwggEFMB0GA1UdDgQWBBSlGDNTP5VgoUMW747G
// SIG // r9Irup5Y0DAfBgNVHSMEGDAWgBQjNPjZUkZwCu1A+3b7
// SIG // syuwwzWzDzBUBgNVHR8ETTBLMEmgR6BFhkNodHRwOi8v
// SIG // Y3JsLm1pY3Jvc29mdC5jb20vcGtpL2NybC9wcm9kdWN0
// SIG // cy9NaWNyb3NvZnRUaW1lU3RhbXBQQ0EuY3JsMFgGCCsG
// SIG // AQUFBwEBBEwwSjBIBggrBgEFBQcwAoY8aHR0cDovL3d3
// SIG // dy5taWNyb3NvZnQuY29tL3BraS9jZXJ0cy9NaWNyb3Nv
// SIG // ZnRUaW1lU3RhbXBQQ0EuY3J0MBMGA1UdJQQMMAoGCCsG
// SIG // AQUFBwMIMA0GCSqGSIb3DQEBBQUAA4IBAQB+zLB75S++
// SIG // 51a1z3PbqlLRFjnGtM361/4eZbXnSPObRogFZmomhl7+
// SIG // h1jcxmOOOID0CEZ8K3OxDr9BqsvHqpSkN/BkOeHF1fnO
// SIG // B86r5CXwaa7URuL+ZjI815fFMiH67holoF4MQiwRMzqC
// SIG // g/3tHbO+zpGkkSVxuatysJ6v5M8AYolwqbhKUIzuLyJk
// SIG // pajmTWuVLBx57KejMdqQYJCkbv6TAg0/LCQNxmomgVGD
// SIG // ShC7dWNEqmkIxgPr4s8L7VY67O9ypwoM9ADTIrivInKz
// SIG // 58ScCyiggMrj4dc5ZjDnRhcY5/qC+lkLeryoDf4c/wOL
// SIG // Y7JNEgIjTy2zhYQ74qFH6M8VMIIE7DCCA9SgAwIBAgIT
// SIG // MwAAALARrwqL0Duf3QABAAAAsDANBgkqhkiG9w0BAQUF
// SIG // ADB5MQswCQYDVQQGEwJVUzETMBEGA1UECBMKV2FzaGlu
// SIG // Z3RvbjEQMA4GA1UEBxMHUmVkbW9uZDEeMBwGA1UEChMV
// SIG // TWljcm9zb2Z0IENvcnBvcmF0aW9uMSMwIQYDVQQDExpN
// SIG // aWNyb3NvZnQgQ29kZSBTaWduaW5nIFBDQTAeFw0xMzAx
// SIG // MjQyMjMzMzlaFw0xNDA0MjQyMjMzMzlaMIGDMQswCQYD
// SIG // VQQGEwJVUzETMBEGA1UECBMKV2FzaGluZ3RvbjEQMA4G
// SIG // A1UEBxMHUmVkbW9uZDEeMBwGA1UEChMVTWljcm9zb2Z0
// SIG // IENvcnBvcmF0aW9uMQ0wCwYDVQQLEwRNT1BSMR4wHAYD
// SIG // VQQDExVNaWNyb3NvZnQgQ29ycG9yYXRpb24wggEiMA0G
// SIG // CSqGSIb3DQEBAQUAA4IBDwAwggEKAoIBAQDor1yiIA34
// SIG // KHy8BXt/re7rdqwoUz8620B9s44z5lc/pVEVNFSlz7SL
// SIG // qT+oN+EtUO01Fk7vTXrbE3aIsCzwWVyp6+HXKXXkG4Un
// SIG // m/P4LZ5BNisLQPu+O7q5XHWTFlJLyjPFN7Dz636o9UEV
// SIG // XAhlHSE38Cy6IgsQsRCddyKFhHxPuRuQsPWj/ov0DJpO
// SIG // oPXJCiHiquMBNkf9L4JqgQP1qTXclFed+0vUDoLbOI8S
// SIG // /uPWenSIZOFixCUuKq6dGB8OHrbCryS0DlC83hyTXEmm
// SIG // ebW22875cHsoAYS4KinPv6kFBeHgD3FN/a1cI4Mp68fF
// SIG // SsjoJ4TTfsZDC5UABbFPZXHFAgMBAAGjggFgMIIBXDAT
// SIG // BgNVHSUEDDAKBggrBgEFBQcDAzAdBgNVHQ4EFgQUWXGm
// SIG // WjNN2pgHgP+EHr6H+XIyQfIwUQYDVR0RBEowSKRGMEQx
// SIG // DTALBgNVBAsTBE1PUFIxMzAxBgNVBAUTKjMxNTk1KzRm
// SIG // YWYwYjcxLWFkMzctNGFhMy1hNjcxLTc2YmMwNTIzNDRh
// SIG // ZDAfBgNVHSMEGDAWgBTLEejK0rQWWAHJNy4zFha5TJoK
// SIG // HzBWBgNVHR8ETzBNMEugSaBHhkVodHRwOi8vY3JsLm1p
// SIG // Y3Jvc29mdC5jb20vcGtpL2NybC9wcm9kdWN0cy9NaWND
// SIG // b2RTaWdQQ0FfMDgtMzEtMjAxMC5jcmwwWgYIKwYBBQUH
// SIG // AQEETjBMMEoGCCsGAQUFBzAChj5odHRwOi8vd3d3Lm1p
// SIG // Y3Jvc29mdC5jb20vcGtpL2NlcnRzL01pY0NvZFNpZ1BD
// SIG // QV8wOC0zMS0yMDEwLmNydDANBgkqhkiG9w0BAQUFAAOC
// SIG // AQEAMdduKhJXM4HVncbr+TrURE0Inu5e32pbt3nPApy8
// SIG // dmiekKGcC8N/oozxTbqVOfsN4OGb9F0kDxuNiBU6fNut
// SIG // zrPJbLo5LEV9JBFUJjANDf9H6gMH5eRmXSx7nR2pEPoc
// SIG // sHTyT2lrnqkkhNrtlqDfc6TvahqsS2Ke8XzAFH9IzU2y
// SIG // RPnwPJNtQtjofOYXoJtoaAko+QKX7xEDumdSrcHps3Om
// SIG // 0mPNSuI+5PNO/f+h4LsCEztdIN5VP6OukEAxOHUoXgSp
// SIG // Rm3m9Xp5QL0fzehF1a7iXT71dcfmZmNgzNWahIeNJDD3
// SIG // 7zTQYx2xQmdKDku/Og7vtpU6pzjkJZIIpohmgjCCBbww
// SIG // ggOkoAMCAQICCmEzJhoAAAAAADEwDQYJKoZIhvcNAQEF
// SIG // BQAwXzETMBEGCgmSJomT8ixkARkWA2NvbTEZMBcGCgmS
// SIG // JomT8ixkARkWCW1pY3Jvc29mdDEtMCsGA1UEAxMkTWlj
// SIG // cm9zb2Z0IFJvb3QgQ2VydGlmaWNhdGUgQXV0aG9yaXR5
// SIG // MB4XDTEwMDgzMTIyMTkzMloXDTIwMDgzMTIyMjkzMlow
// SIG // eTELMAkGA1UEBhMCVVMxEzARBgNVBAgTCldhc2hpbmd0
// SIG // b24xEDAOBgNVBAcTB1JlZG1vbmQxHjAcBgNVBAoTFU1p
// SIG // Y3Jvc29mdCBDb3Jwb3JhdGlvbjEjMCEGA1UEAxMaTWlj
// SIG // cm9zb2Z0IENvZGUgU2lnbmluZyBQQ0EwggEiMA0GCSqG
// SIG // SIb3DQEBAQUAA4IBDwAwggEKAoIBAQCycllcGTBkvx2a
// SIG // YCAgQpl2U2w+G9ZvzMvx6mv+lxYQ4N86dIMaty+gMuz/
// SIG // 3sJCTiPVcgDbNVcKicquIEn08GisTUuNpb15S3GbRwfa
// SIG // /SXfnXWIz6pzRH/XgdvzvfI2pMlcRdyvrT3gKGiXGqel
// SIG // cnNW8ReU5P01lHKg1nZfHndFg4U4FtBzWwW6Z1KNpbJp
// SIG // L9oZC/6SdCnidi9U3RQwWfjSjWL9y8lfRjFQuScT5EAw
// SIG // z3IpECgixzdOPaAyPZDNoTgGhVxOVoIoKgUyt0vXT2Pn
// SIG // 0i1i8UU956wIAPZGoZ7RW4wmU+h6qkryRs83PDietHdc
// SIG // pReejcsRj1Y8wawJXwPTAgMBAAGjggFeMIIBWjAPBgNV
// SIG // HRMBAf8EBTADAQH/MB0GA1UdDgQWBBTLEejK0rQWWAHJ
// SIG // Ny4zFha5TJoKHzALBgNVHQ8EBAMCAYYwEgYJKwYBBAGC
// SIG // NxUBBAUCAwEAATAjBgkrBgEEAYI3FQIEFgQU/dExTtMm
// SIG // ipXhmGA7qDFvpjy82C0wGQYJKwYBBAGCNxQCBAweCgBT
// SIG // AHUAYgBDAEEwHwYDVR0jBBgwFoAUDqyCYEBWJ5flJRP8
// SIG // KuEKU5VZ5KQwUAYDVR0fBEkwRzBFoEOgQYY/aHR0cDov
// SIG // L2NybC5taWNyb3NvZnQuY29tL3BraS9jcmwvcHJvZHVj
// SIG // dHMvbWljcm9zb2Z0cm9vdGNlcnQuY3JsMFQGCCsGAQUF
// SIG // BwEBBEgwRjBEBggrBgEFBQcwAoY4aHR0cDovL3d3dy5t
// SIG // aWNyb3NvZnQuY29tL3BraS9jZXJ0cy9NaWNyb3NvZnRS
// SIG // b290Q2VydC5jcnQwDQYJKoZIhvcNAQEFBQADggIBAFk5
// SIG // Pn8mRq/rb0CxMrVq6w4vbqhJ9+tfde1MOy3XQ60L/svp
// SIG // LTGjI8x8UJiAIV2sPS9MuqKoVpzjcLu4tPh5tUly9z7q
// SIG // QX/K4QwXaculnCAt+gtQxFbNLeNK0rxw56gNogOlVuC4
// SIG // iktX8pVCnPHz7+7jhh80PLhWmvBTI4UqpIIck+KUBx3y
// SIG // 4k74jKHK6BOlkU7IG9KPcpUqcW2bGvgc8FPWZ8wi/1wd
// SIG // zaKMvSeyeWNWRKJRzfnpo1hW3ZsCRUQvX/TartSCMm78
// SIG // pJUT5Otp56miLL7IKxAOZY6Z2/Wi+hImCWU4lPF6H0q7
// SIG // 0eFW6NB4lhhcyTUWX92THUmOLb6tNEQc7hAVGgBd3TVb
// SIG // Ic6YxwnuhQ6MT20OE049fClInHLR82zKwexwo1eSV32U
// SIG // jaAbSANa98+jZwp0pTbtLS8XyOZyNxL0b7E8Z4L5UrKN
// SIG // MxZlHg6K3RDeZPRvzkbU0xfpecQEtNP7LN8fip6sCvsT
// SIG // J0Ct5PnhqX9GuwdgR2VgQE6wQuxO7bN2edgKNAltHIAx
// SIG // H+IOVN3lofvlRxCtZJj/UBYufL8FIXrilUEnacOTj5XJ
// SIG // jdibIa4NXJzwoq6GaIMMai27dmsAHZat8hZ79haDJLmI
// SIG // z2qoRzEvmtzjcT3XAH5iR9HOiMm4GPoOco3Boz2vAkBq
// SIG // /2mbluIQqBC0N1AI1sM9MIIGBzCCA++gAwIBAgIKYRZo
// SIG // NAAAAAAAHDANBgkqhkiG9w0BAQUFADBfMRMwEQYKCZIm
// SIG // iZPyLGQBGRYDY29tMRkwFwYKCZImiZPyLGQBGRYJbWlj
// SIG // cm9zb2Z0MS0wKwYDVQQDEyRNaWNyb3NvZnQgUm9vdCBD
// SIG // ZXJ0aWZpY2F0ZSBBdXRob3JpdHkwHhcNMDcwNDAzMTI1
// SIG // MzA5WhcNMjEwNDAzMTMwMzA5WjB3MQswCQYDVQQGEwJV
// SIG // UzETMBEGA1UECBMKV2FzaGluZ3RvbjEQMA4GA1UEBxMH
// SIG // UmVkbW9uZDEeMBwGA1UEChMVTWljcm9zb2Z0IENvcnBv
// SIG // cmF0aW9uMSEwHwYDVQQDExhNaWNyb3NvZnQgVGltZS1T
// SIG // dGFtcCBQQ0EwggEiMA0GCSqGSIb3DQEBAQUAA4IBDwAw
// SIG // ggEKAoIBAQCfoWyx39tIkip8ay4Z4b3i48WZUSNQrc7d
// SIG // GE4kD+7Rp9FMrXQwIBHrB9VUlRVJlBtCkq6YXDAm2gBr
// SIG // 6Hu97IkHD/cOBJjwicwfyzMkh53y9GccLPx754gd6udO
// SIG // o6HBI1PKjfpFzwnQXq/QsEIEovmmbJNn1yjcRlOwhtDl
// SIG // KEYuJ6yGT1VSDOQDLPtqkJAwbofzWTCd+n7Wl7PoIZd+
// SIG // +NIT8wi3U21StEWQn0gASkdmEScpZqiX5NMGgUqi+YSn
// SIG // EUcUCYKfhO1VeP4Bmh1QCIUAEDBG7bfeI0a7xC1Un68e
// SIG // eEExd8yb3zuDk6FhArUdDbH895uyAc4iS1T/+QXDwiAL
// SIG // AgMBAAGjggGrMIIBpzAPBgNVHRMBAf8EBTADAQH/MB0G
// SIG // A1UdDgQWBBQjNPjZUkZwCu1A+3b7syuwwzWzDzALBgNV
// SIG // HQ8EBAMCAYYwEAYJKwYBBAGCNxUBBAMCAQAwgZgGA1Ud
// SIG // IwSBkDCBjYAUDqyCYEBWJ5flJRP8KuEKU5VZ5KShY6Rh
// SIG // MF8xEzARBgoJkiaJk/IsZAEZFgNjb20xGTAXBgoJkiaJ
// SIG // k/IsZAEZFgltaWNyb3NvZnQxLTArBgNVBAMTJE1pY3Jv
// SIG // c29mdCBSb290IENlcnRpZmljYXRlIEF1dGhvcml0eYIQ
// SIG // ea0WoUqgpa1Mc1j0BxMuZTBQBgNVHR8ESTBHMEWgQ6BB
// SIG // hj9odHRwOi8vY3JsLm1pY3Jvc29mdC5jb20vcGtpL2Ny
// SIG // bC9wcm9kdWN0cy9taWNyb3NvZnRyb290Y2VydC5jcmww
// SIG // VAYIKwYBBQUHAQEESDBGMEQGCCsGAQUFBzAChjhodHRw
// SIG // Oi8vd3d3Lm1pY3Jvc29mdC5jb20vcGtpL2NlcnRzL01p
// SIG // Y3Jvc29mdFJvb3RDZXJ0LmNydDATBgNVHSUEDDAKBggr
// SIG // BgEFBQcDCDANBgkqhkiG9w0BAQUFAAOCAgEAEJeKw1wD
// SIG // RDbd6bStd9vOeVFNAbEudHFbbQwTq86+e4+4LtQSooxt
// SIG // YrhXAstOIBNQmd16QOJXu69YmhzhHQGGrLt48ovQ7DsB
// SIG // 7uK+jwoFyI1I4vBTFd1Pq5Lk541q1YDB5pTyBi+FA+mR
// SIG // KiQicPv2/OR4mS4N9wficLwYTp2OawpylbihOZxnLcVR
// SIG // DupiXD8WmIsgP+IHGjL5zDFKdjE9K3ILyOpwPf+FChPf
// SIG // wgphjvDXuBfrTot/xTUrXqO/67x9C0J71FNyIe4wyrt4
// SIG // ZVxbARcKFA7S2hSY9Ty5ZlizLS/n+YWGzFFW6J1wlGys
// SIG // OUzU9nm/qhh6YinvopspNAZ3GmLJPR5tH4LwC8csu89D
// SIG // s+X57H2146SodDW4TsVxIxImdgs8UoxxWkZDFLyzs7BN
// SIG // Z8ifQv+AeSGAnhUwZuhCEl4ayJ4iIdBD6Svpu/RIzCzU
// SIG // 2DKATCYqSCRfWupW76bemZ3KOm+9gSd0BhHudiG/m4LB
// SIG // J1S2sWo9iaF2YbRuoROmv6pH8BJv/YoybLL+31HIjCPJ
// SIG // Zr2dHYcSZAI9La9Zj7jkIeW1sMpjtHhUBdRBLlCslLCl
// SIG // eKuzoJZ1GtmShxN1Ii8yqAhuoFuMJb+g74TKIdbrHk/J
// SIG // mu5J4PcBZW+JC33Iacjmbuqnl84xKf8OxVtc2E0bodj6
// SIG // L54/LlUWa8kTo/0xggSFMIIEgQIBATCBkDB5MQswCQYD
// SIG // VQQGEwJVUzETMBEGA1UECBMKV2FzaGluZ3RvbjEQMA4G
// SIG // A1UEBxMHUmVkbW9uZDEeMBwGA1UEChMVTWljcm9zb2Z0
// SIG // IENvcnBvcmF0aW9uMSMwIQYDVQQDExpNaWNyb3NvZnQg
// SIG // Q29kZSBTaWduaW5nIFBDQQITMwAAALARrwqL0Duf3QAB
// SIG // AAAAsDAJBgUrDgMCGgUAoIGeMBkGCSqGSIb3DQEJAzEM
// SIG // BgorBgEEAYI3AgEEMBwGCisGAQQBgjcCAQsxDjAMBgor
// SIG // BgEEAYI3AgEVMCMGCSqGSIb3DQEJBDEWBBS5KPcaf8Mr
// SIG // AdASPRNAq4W79PDJXDA+BgorBgEEAYI3AgEMMTAwLqAU
// SIG // gBIAcgBlAG0AbwB0AGUALgBqAHOhFoAUaHR0cDovL21p
// SIG // Y3Jvc29mdC5jb20wDQYJKoZIhvcNAQEBBQAEggEAHicE
// SIG // xZbePJSAE3ntnpBkKxV7SwUXjfTdTYrKGxz6gDsD6FqL
// SIG // iWQxVw1I9dqjsADAg6UFCCAjIilnWzvRod3YN8soSm8k
// SIG // ip0ulO3mUfv5UTIYUCixu2++1IpqWmUlb5kYo7cmG23z
// SIG // A+rJTjo1ELFu01arC9AaPt0Np4fhGiBsB4Do5HtpmgZc
// SIG // NKRzWngh7NP9HKx/zSyx6gvFg7cHODp95Km7WP0nanti
// SIG // xqDH3k+Z24PLju9dQ1ecNELInGRchM0b93hS2hZfd7Jr
// SIG // eJB1jlHAWurXZ9P6YuP+v8wGg1dzqvMj8sJ3W82S+3W5
// SIG // 39JnxIHsqskAJMgg5R9r5OKetsRuQKGCAigwggIkBgkq
// SIG // hkiG9w0BCQYxggIVMIICEQIBATCBjjB3MQswCQYDVQQG
// SIG // EwJVUzETMBEGA1UECBMKV2FzaGluZ3RvbjEQMA4GA1UE
// SIG // BxMHUmVkbW9uZDEeMBwGA1UEChMVTWljcm9zb2Z0IENv
// SIG // cnBvcmF0aW9uMSEwHwYDVQQDExhNaWNyb3NvZnQgVGlt
// SIG // ZS1TdGFtcCBQQ0ECEzMAAAArOTJIwbLJSPMAAAAAACsw
// SIG // CQYFKw4DAhoFAKBdMBgGCSqGSIb3DQEJAzELBgkqhkiG
// SIG // 9w0BBwEwHAYJKoZIhvcNAQkFMQ8XDTEzMDMxNTA2MzQw
// SIG // NFowIwYJKoZIhvcNAQkEMRYEFCng8QULr5DVOhlN5nit
// SIG // ohAGFqfYMA0GCSqGSIb3DQEBBQUABIIBADI7IFYbubeG
// SIG // AV4HG51/dcW8myia5b2qiFHRpeL2/4lheIKK0ZJAFmsx
// SIG // s2wziDZUUQjTWHrkooRp9hbcWyivuf59ftGiZ5uiaU4n
// SIG // /yRKmfgLhsV4ypdGheY0W2Iy9jwKEhUaTBNZjk+jeQIB
// SIG // ReOKQsWMHKCWV0UpPV6qGYDmBm9aKeiWZ9uQERIwLjFf
// SIG // 95lDkmk1K6MMBj3h/EPs9RtZFth0ymzrE7bah7UKQI85
// SIG // PQeM0UmwYUxFcuPt/5mBGWF5S9lfgF3b6LDE8ylFaZyW
// SIG // 0aEMR46hdrTdppRgU520Clh6SUWTuY+JSVkujBil0nm8
// SIG // +eh4tcdKAOBkklO/t9cNCTE=
// SIG // End signature block
